// The program (sieve of erastophanes):
// #define mem[0] i
// #define mem[1] j
// for (i = 2; i < 1024; i++) mem[i] = i;
// i = 1; // So the scan will catch 2. mem[1] will be ignored.
// while (1) {
//   // Scan to next prime
//   do {i++; if (i == 1024) goto done; } while (mem[i] == 0);
//   // i is now prime. Zero all of its multiples.
//   for (j = 2*i; j < 1024; j+=i) mem[j] = 0;
// }
// done: halt
#include <stdint.h>
#include <stdlib.h>
#include <iostream>
#include <iomanip>

typedef uint64_t addr_t;
typedef uint32_t word_t;

class Op;

class Machine {
public:
  bool running;

  word_t acc;        // Accumulator
  word_t mem[1<<26]; // Memory

  Op *next; // Pointer to the next operation.

  Machine() : running(true) { for(unsigned i = 0; i < 1<<26; i++) mem[i] = 0; }

  void show_state() {
    using std::cout; using std::endl;    using std::hex; 
    using std::setw; using std::setfill; using std::dec;

    for (unsigned start_i = 0; start_i < 64; start_i += 8) {
      cout << "0x" << hex << setw(4) << setfill('0') << start_i << ':';
      for (unsigned i = start_i; i < start_i + 8; i++) {
        cout << dec << ' ' << setw(4) << setfill('0') << mem[i];
      } 
      cout << endl;
    }
    cout << endl;
  }
  
};

// Dispatch table and micro-op impementations.
extern "C" {
  Machine *__m;

  void __halt           (uint64_t)    { __m->running = false;                }
  void __load           (uint64_t arg){ __m->acc = __m->mem[arg];            }
  void __indirect_load  (uint64_t arg){ __m->acc = __m->mem[__m->mem[arg]];  }
  void __store          (uint64_t arg){ __m->mem[arg] = __m->acc;            }
  void __indirect_store (uint64_t arg){ __m->mem[__m->mem[arg]] = __m->acc;  }
  void __branch_always  (uint64_t arg){ __m->next = (Op*)arg;                }
  void __branch_zero    (uint64_t arg){if(__m->acc==0) __m->next = (Op*)arg; }
  void __branch_not_zero(uint64_t arg){if(__m->acc!=0) __m->next = (Op*)arg; }
  void __mem_add        (uint64_t arg){ __m->acc += __m->mem[arg];           }
  void __mem_sum        (uint64_t arg){ __m->acc -= __m->mem[arg];           }
  void __mem_nand       (uint64_t arg){ __m->acc = ~(__m->acc&__m->mem[arg]);}
  void __imm_add        (uint64_t arg){ __m->acc += arg;                     }
  void __imm_sub        (uint64_t arg){ __m->acc -= arg;                     }
  void __imm_nand       (uint64_t arg){ __m->acc = ~(__m->acc&arg);          }
  void __imm_load       (uint64_t arg){ __m->acc = arg;                      }

  typedef void (*op_type)(uint64_t);
};

class Op {
public:
  op_type  type;
  uint64_t arg;

  Op(op_type o)             : type(o) {}
  Op(op_type o, word_t arg) : type(o), arg(uint64_t(arg)) {}
  Op(op_type o, Op& target) : type(o), arg(uint64_t(&target)) {}

  virtual ~Op() {}
};

// Initialize the program and run it.
int main() {

  Machine *m = new Machine();

  Op program[] = {
    Op(__imm_load,       2),             //  0
    Op(__store,          0),             //  1

    Op(__indirect_store, 0),             //  2 INIT1
    Op(__imm_sub,        (1<<26)-1),     //  3
    Op(__branch_zero,    program[9]),    //  4
    Op(__load,           0),             //  5
    Op(__imm_add,        1),             //  6
    Op(__store,          0),             //  7
    Op(__branch_always,  program[2]),    //  8 

    Op(__imm_load,       1),             //  9 INIT2
    Op(__store,          0),             // 10

    Op(__load,           0),             // 11 MAIN
    Op(__imm_add,        1),             // 12
    Op(__store,          0),             // 13
    Op(__imm_sub,        (1<<26)),       // 14
    Op(__branch_zero,    program[30]),   // 15
    Op(__indirect_load,  0),             // 16
    Op(__branch_zero,    program[11]),   // 17
    Op(__load,           0),             // 18
    Op(__store,          1),             // 19

      Op(__load,           0),           // 20 ZERO
      Op(__mem_add,        1),           // 21
      Op(__store,          1),           // 22
      Op(__imm_sub,        1<<26),       // 23
      Op(__imm_nand,       0x80000000),  // 24
      Op(__imm_nand,       0xffffffff),  // 25
      Op(__branch_zero,    program[11]), // 26

      Op(__imm_load,       0),           // 27
      Op(__indirect_store, 1),           // 28
      Op(__branch_always,  program[20]), // 29

    Op(__halt),                          // 30 DONE
  };

  //for (unsigned i = 0; i < 31; i++) {
  //  program[i].set_successor(&program[i+1]);
  //}

  uint64_t icount = 0;
  m->next = &program[0];
  __m = m;
  while (m->running) {
    Op *current = m->next;
    m->next = current+1;
    current->type(current->arg);
    if ((++icount)%100000000 == 0) {std::cout << "Inst " << icount << ".\n";}
  }

  std::cout << icount << " total instructions.\n";

  m->show_state();

  return 0;
}
