// The program (sieve of erastophanes):
// #define mem[0] i
// #define mem[1] j
// for (i = 2; i < 1024; i++) mem[i] = i;
// i = 1; // So the scan will catch 2. mem[1] will be ignored.
// while (1) {
//   // Scan to next prime
//   do {i++; if (i == 1024) goto done; } while (mem[i] == 0);
//   // i is now prime. Zero all of its multiples.
//   for (j = 2*i; j < 1024; j+=i) mem[j] = 0;
// }
// done: halt
#include <stdint.h>
#include <stdlib.h>
#include <iostream>
#include <iomanip>

typedef uint64_t addr_t;
typedef uint32_t word_t;

class Op;

class Machine {
public:
  bool running;

  word_t acc;        // Accumulator
  word_t mem[1<<26]; // Memory

  Op *next; // Pointer to the next operation.

  Machine() : running(true) { for(unsigned i = 0; i < 1<<26; i++) mem[i] = 0; }

  void show_state() {
    using std::cout; using std::endl;    using std::hex; 
    using std::setw; using std::setfill; using std::dec;

    for (unsigned start_i = 0; start_i < 128; start_i += 8) {
      cout << "0x" << hex << setw(4) << setfill('0') << start_i << ':';
      for (unsigned i = start_i; i < start_i + 8; i++) {
        cout << dec << ' ' << setw(4) << setfill('0') << mem[i];
      } 
      cout << endl;
    }
    cout << endl;
  }
  
};

class Op {
public:
  Op() {}
  virtual ~Op() {}
  virtual void execute(Machine& m) = 0;

  Op *succ;
  void set_successor(Op *s) { succ = s; }
  Op *successor() const { return succ; }
};

// Halt
class HaltOp : public Op { public: 
  void execute(Machine& m) { m.running = false; }
};

// Load: acc = mem[addr];
class LoadOp : public Op {
public:
  word_t addr;
  LoadOp(word_t addr) : addr(addr) {}
  void execute(Machine &m) { m.acc = m.mem[addr]; }
};

// Indirect load: mem[addr] = addr;
class IndirectLoadOp : public Op {
public:
  word_t addr;
  IndirectLoadOp(word_t addr) : addr(addr) {}
  void execute(Machine &m) { m.acc = m.mem[m.mem[addr]]; }
};

// Store: mem[addr] = acc;
class StoreOp : public Op {
public: 
  word_t addr;
  StoreOp(word_t addr) : addr(addr) {}
  void execute(Machine &m) { m.mem[addr] = m.acc; }
};

// Indirect store: mem[mem[addr]] = acc;
class IndirectStoreOp : public Op {
public:
  word_t addr;
  IndirectStoreOp(word_t addr) : addr(addr) {}
  void execute(Machine &m) { m.mem[m.mem[addr]] = m.acc; }
};

// Control flow operation.
class ControlOp : public Op {
public:
  Op *target;
  ControlOp(Op &target) : target(&target) {}
  virtual ~ControlOp() {}
  virtual bool take(word_t acc) = 0;
  void execute(Machine &m) { if (take(m.acc)) m.next = target; }
};

class BranchAlways : public ControlOp {
public:
  BranchAlways(Op &target) : ControlOp(target) {}
  bool take(word_t acc) { return true; }
};

class BranchZero : public ControlOp {
public:
  BranchZero(Op &target) : ControlOp(target) {}
  bool take(word_t acc) { return acc == 0; }
};

class BranchNotZero : public ControlOp {
public:
  BranchNotZero(Op &target) : ControlOp(target) {}
  bool take(word_t acc) { return acc != 0; }
};

// Immediate operation: acc = ImmOp(acc, imm);
//   'imm' can be an address or data.
class ImmOp : public Op {
public:
  word_t imm;
  ImmOp(word_t imm) : imm(imm) {}
  virtual ~ImmOp() {}
  virtual void execute(Machine &m) = 0;
};

// Memory arithmetic operations: acc = [op](acc, *imm)
class MemArithOp : public ImmOp {
public:
  MemArithOp(word_t imm) : ImmOp(imm) {}
  virtual ~MemArithOp() {}

  virtual word_t compute(word_t acc, word_t val) = 0;

  virtual void execute(Machine &m) { m.acc = compute(m.acc, m.mem[imm]); }
};

class MemAdd : public MemArithOp {
public:
  MemAdd(word_t imm) : MemArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return acc + val; }
};

class MemSub : public MemArithOp {
public: 
  MemSub(word_t imm) : MemArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return acc - val; }
};

class MemNand : public MemArithOp {
public:
  MemNand(word_t imm) : MemArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return ~(acc & val); }
};

// Immediate arithmetic operations: acc = [op](acc, imm)
class ImmArithOp : public ImmOp {
public:
  ImmArithOp(word_t imm) : ImmOp(imm) {}
  virtual ~ImmArithOp() {}

  virtual word_t compute(word_t acc, word_t val) = 0;
  virtual void execute(Machine &m) { m.acc = compute(m.acc, imm); }
};

class ImmLoad : public ImmArithOp {
public:
  ImmLoad(word_t imm) : ImmArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return val; }
};

class ImmAdd : public ImmArithOp {
public:
  ImmAdd(word_t imm) : ImmArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return acc + val; }
};

class ImmSub : public ImmArithOp {
public:
  ImmSub(word_t imm) : ImmArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return acc - val; }
};

class ImmNand : public ImmArithOp {
public:
  ImmNand(word_t imm) : ImmArithOp(imm) {}
  word_t compute(word_t acc, word_t val) { return ~(acc & val); }
};

// Initialize the program and run it.
int main() {

  Machine *m = new Machine();

  Op *program[] = {
    new ImmLoad(2),                            //  0
    new StoreOp(0),                            //  1
    new IndirectStoreOp(0), /*INIT1*/          //  2 INIT1
    new ImmSub((1<<26)-1),                     //  3
    0, //new BranchZero(*program[9]/*INIT2*/), //  4
    new LoadOp(0),                             //  5
    new ImmAdd(1),                             //  6
    new StoreOp(0),                            //  7
    new BranchAlways(*program[2]/*INIT1*/),    //  8 
    new ImmLoad(1), /*INIT2*/                  //  9 INIT2
    new StoreOp(0),                            // 10
    new LoadOp(0), /*MAIN*/                    // 11 MAIN
    new ImmAdd(1),                             // 12
    new StoreOp(0),                            // 13
    new ImmSub(1<<26),                         // 14
    0, //new BranchZero(*program[29]/*DONE*/), // 15
    new IndirectLoadOp(0),                     // 16
    new BranchZero(*program[11]/*MAIN*/),      // 17
    new LoadOp(0),                             // 18
    new StoreOp(1),                            // 19
    new LoadOp(0), /*ZERO*/                    // 20 ZERO
    new MemAdd(1),                             // 21
    new StoreOp(1),                            // 22
    new ImmSub(1<<26),                         // 23
    new ImmNand(0x80000000),                   // 24
    new ImmNand(0xffffffff),                   // 25
    new BranchZero(*program[11]/*MAIN*/),      // 26
    new ImmLoad(0),                            // 27
    new IndirectStoreOp(1),                    // 28
    new BranchAlways(*program[20]/*ZERO*/),    // 29
    new HaltOp(),                              // 30 DONE
    0
  };

  // Patch up forward branches.
  program[4]  = new BranchZero(*program[9] );
  program[15] = new BranchZero(*program[30]);

  for (unsigned i = 0; program[i]; i++) {
    program[i]->set_successor(program[i+1]);
  }

  uint64_t icount = 0;
  m->next = program[0];
  while (m->running) {
    Op *current = m->next;
    m->next = current->successor();
    current->execute(*m);

    icount += 1;
    if (icount%100000000 == 0) {std::cout << "Inst " << icount << ".\n";}
  }

  std::cout << icount << " total instructions.\n";

  m->show_state();

  return 0;
}
