// The program (sieve of erastophanes):
// #define mem[0] i
// #define mem[1] j
// for (i = 2; i < 1024; i++) mem[i] = i;
// i = 1; // So the scan will catch 2. mem[1] will be ignored.
// while (1) {
//   // Scan to next prime
//   do {i++; if (i == 1024) goto done; } while (mem[i] == 0);
//   // i is now prime. Zero all of its multiples.
//   for (j = 2*i; j < 1024; j+=i) mem[j] = 0;
// }
// done: halt
#include <stdint.h>
#include <stdlib.h>
#include <iostream>
#include <iomanip>

typedef uint64_t addr_t;
typedef uint32_t word_t;

class Op;

class Machine {
public:
  bool running;

  word_t acc;        // Accumulator
  word_t mem[1<<26]; // Memory

  Op *next; // Pointer to the next operation.

  Machine() : running(true) { for(unsigned i = 0; i < 1<<26; i++) mem[i] = 0; }

  void show_state() {
    using std::cout; using std::endl;    using std::hex; 
    using std::setw; using std::setfill; using std::dec;

    for (unsigned start_i = 0; start_i < 64; start_i += 8) {
      cout << "0x" << hex << setw(4) << setfill('0') << start_i << ':';
      for (unsigned i = start_i; i < start_i + 8; i++) {
        cout << dec << ' ' << setw(4) << setfill('0') << mem[i];
      } 
      cout << endl;
    }
    cout << endl;
  }
  
};

enum op_type {
  HALT, LOAD, INDIRECT_LOAD, STORE, INDIRECT_STORE, BRANCH_ALWAYS, BRANCH_ZERO,
  BRANCH_NOT_ZERO, MEM_ADD, MEM_SUB, MEM_NAND, IMM_ADD, IMM_SUB, IMM_NAND, 
  IMM_LOAD
};

class Op {
public:
  op_type  type;
  Op      *targ;
  word_t   arg;

  Op(op_type o)             : type(o) {}
  Op(op_type o, word_t arg) : type(o), arg(arg) {}
  Op(op_type o, Op& target) : type(o), targ(&target) {}

  virtual ~Op() {}
  void execute(Machine& m) {
    switch (type) {
    case HALT:                            m.running = false;          break;
    case LOAD:                            m.acc = m.mem[arg];         break;
    case INDIRECT_LOAD:                   m.acc = m.mem[m.mem[arg]];  break;
    case STORE:                           m.mem[arg] = m.acc;         break;
    case INDIRECT_STORE:                  m.mem[m.mem[arg]] = m.acc;  break;
    case BRANCH_ALWAYS:                   m.next = targ;              break;
    case BRANCH_ZERO:     if (m.acc == 0) m.next = targ;              break;
    case BRANCH_NOT_ZERO: if (m.acc != 0) m.next = targ;              break;
    case MEM_ADD:                         m.acc += m.mem[arg];        break;
    case MEM_SUB:                         m.acc -= m.mem[arg];        break;
    case MEM_NAND:                        m.acc =~(m.acc&m.mem[arg]); break;
    case IMM_ADD:                         m.acc += arg;               break;
    case IMM_SUB:                         m.acc -= arg;               break;
    case IMM_NAND:                        m.acc = ~(m.acc & arg);     break;
    case IMM_LOAD:                        m.acc = arg;                break;
    }
  }

  //Op *succ;
  //
  //void set_successor(Op *s) { succ = s; }
  //Op *successor() const { return succ; }
};

// Initialize the program and run it.
int main() {

  Machine *m = new Machine();

  Op program[] = {
    Op(IMM_LOAD,       2),             //  0
    Op(STORE,          0),             //  1

    Op(INDIRECT_STORE, 0),             //  2 INIT1
    Op(IMM_SUB,        (1<<26)-1),     //  3
    Op(BRANCH_ZERO,    program[9]),    //  4
    Op(LOAD,           0),             //  5
    Op(IMM_ADD,        1),             //  6
    Op(STORE,          0),             //  7
    Op(BRANCH_ALWAYS,  program[2]),    //  8 

    Op(IMM_LOAD,       1),             //  9 INIT2
    Op(STORE,          0),             // 10

    Op(LOAD,           0),             // 11 MAIN
    Op(IMM_ADD,        1),             // 12
    Op(STORE,          0),             // 13
    Op(IMM_SUB,        (1<<26)),       // 14
    Op(BRANCH_ZERO,    program[30]),   // 15
    Op(INDIRECT_LOAD,  0),             // 16
    Op(BRANCH_ZERO,    program[11]),   // 17
    Op(LOAD,           0),             // 18
    Op(STORE,          1),             // 19

      Op(LOAD,           0),           // 20 ZERO
      Op(MEM_ADD,        1),           // 21
      Op(STORE,          1),           // 22
      Op(IMM_SUB,        1<<26),       // 23
      Op(IMM_NAND,       0x80000000),  // 24
      Op(IMM_NAND,       0xffffffff),  // 25
      Op(BRANCH_ZERO,    program[11]), // 26

      Op(IMM_LOAD,       0),           // 27
      Op(INDIRECT_STORE, 1),           // 28
      Op(BRANCH_ALWAYS,  program[20]), // 29

    Op(HALT),                          // 30 DONE
  };

  //for (unsigned i = 0; i < 31; i++) {
  //  program[i].set_successor(&program[i+1]);
  //}

  uint64_t icount = 0;
  m->next = &program[0];
  while (m->running) {
    Op *current = m->next;
    m->next = current+1;
    current->execute(*m);
    if ((++icount)%100000000 == 0) {std::cout << "Inst " << icount << ".\n";}
  }

  std::cout << icount << " total instructions.\n";

  m->show_state();

  return 0;
}
